import type * as msalBrowser from "@azure/msal-browser";
import type { AccessToken, GetTokenOptions } from "@azure/core-auth";
import type { AuthenticationRecord, MsalResult } from "../types.js";
import type { CredentialLogger } from "../../util/logging.js";
import type { MsalFlow, MsalFlowOptions } from "./flows.js";
import type { BrowserLoginStyle } from "../../credentials/interactiveBrowserCredentialOptions.js";
import type { CredentialFlowGetTokenOptions } from "../credentials.js";
import type { LogPolicyOptions } from "@azure/core-rest-pipeline";
import type { MultiTenantTokenCredentialOptions } from "../../credentials/multiTenantTokenCredentialOptions.js";
/**
 * Union of the constructor parameters that all MSAL flow types take.
 * Some properties might not be used by some flow types.
 */
export interface MsalBrowserFlowOptions extends MsalFlowOptions {
    tokenCredentialOptions: MultiTenantTokenCredentialOptions;
    redirectUri?: string;
    loginStyle: BrowserLoginStyle;
    loginHint?: string;
    /**
     * Allows users to configure settings for logging policy options, allow logging account information and personally identifiable information for customer support.
     */
    loggingOptions?: LogPolicyOptions & {
        /**
         * Allows logging account information once the authentication flow succeeds.
         */
        allowLoggingAccountIdentifiers?: boolean;
        /**
         * Allows logging personally identifiable information for customer support.
         */
        enableUnsafeSupportLogging?: boolean;
    };
}
/**
 * The common methods we use to work with the MSAL browser flows.
 * @internal
 */
export interface MsalBrowserFlow extends MsalFlow {
    login(scopes?: string[]): Promise<AuthenticationRecord | undefined>;
    handleRedirect(): Promise<AuthenticationRecord | undefined>;
}
/**
 * Generates a MSAL configuration that generally works for browsers
 * @internal
 */
export declare function defaultBrowserMsalConfig(options: MsalBrowserFlowOptions): msalBrowser.Configuration;
/**
 * MSAL partial base client for the browsers.
 *
 * It completes the input configuration with some default values.
 * It also provides with utility protected methods that can be used from any of the clients,
 * which includes handlers for successful responses and errors.
 *
 * @internal
 */
export declare abstract class MsalBrowser implements MsalBrowserFlow {
    protected loginStyle: BrowserLoginStyle;
    protected clientId: string;
    protected tenantId: string;
    protected additionallyAllowedTenantIds: string[];
    protected authorityHost?: string;
    protected account: AuthenticationRecord | undefined;
    protected msalConfig: msalBrowser.Configuration;
    protected disableAutomaticAuthentication?: boolean;
    protected app?: msalBrowser.IPublicClientApplication;
    protected logger: CredentialLogger;
    constructor(options: MsalBrowserFlowOptions);
    /**
     * In the browsers we don't need to init()
     */
    init(): Promise<void>;
    /**
     * Attempts to handle a redirection request the least amount of times possible.
     */
    abstract handleRedirect(): Promise<AuthenticationRecord | undefined>;
    /**
     * Clears MSAL's cache.
     */
    logout(): Promise<void>;
    /**
     * Uses MSAL to retrieve the active account.
     */
    abstract getActiveAccount(): Promise<AuthenticationRecord | undefined>;
    /**
     * Uses MSAL to trigger a redirect or a popup login.
     */
    abstract login(scopes?: string | string[]): Promise<AuthenticationRecord | undefined>;
    /**
     * Attempts to retrieve a token from cache.
     */
    abstract getTokenSilent(scopes: string[]): Promise<AccessToken>;
    /**
     * Attempts to retrieve the token in the browser.
     */
    protected abstract doGetToken(scopes: string[]): Promise<AccessToken>;
    /**
     * Attempts to retrieve an authenticated token from MSAL.
     */
    getToken(scopes: string[], options?: CredentialFlowGetTokenOptions): Promise<AccessToken>;
    /**
     * Handles the MSAL authentication result.
     * If the result has an account, we update the local account reference.
     * If the token received is invalid, an error will be thrown depending on what's missing.
     */
    protected handleResult(scopes: string | string[], result?: MsalResult, getTokenOptions?: GetTokenOptions): AccessToken;
}
//# sourceMappingURL=msalBrowserCommon.d.ts.map