"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.connectInParallel = connectInParallel;
exports.connectInSequence = connectInSequence;
exports.lookupAllAddresses = lookupAllAddresses;
var _net = _interopRequireDefault(require("net"));
var _nodeUrl = _interopRequireDefault(require("node:url"));
var _abortError = _interopRequireDefault(require("./errors/abort-error"));
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
async function connectInParallel(options, lookup, signal) {
  if (signal.aborted) {
    throw new _abortError.default();
  }
  const addresses = await lookupAllAddresses(options.host, lookup, signal);
  return await new Promise((resolve, reject) => {
    const sockets = new Array(addresses.length);
    const errors = [];
    function onError(err) {
      errors.push(err);
      this.removeListener('error', onError);
      this.removeListener('connect', onConnect);
      this.destroy();
      if (errors.length === addresses.length) {
        signal.removeEventListener('abort', onAbort);
        reject(new AggregateError(errors, 'Could not connect (parallel)'));
      }
    }
    function onConnect() {
      signal.removeEventListener('abort', onAbort);
      for (let j = 0; j < sockets.length; j++) {
        const socket = sockets[j];
        if (this === socket) {
          continue;
        }
        socket.removeListener('error', onError);
        socket.removeListener('connect', onConnect);
        socket.destroy();
      }
      resolve(this);
    }
    const onAbort = () => {
      for (let j = 0; j < sockets.length; j++) {
        const socket = sockets[j];
        socket.removeListener('error', onError);
        socket.removeListener('connect', onConnect);
        socket.destroy();
      }
      reject(new _abortError.default());
    };
    for (let i = 0, len = addresses.length; i < len; i++) {
      const socket = sockets[i] = _net.default.connect({
        ...options,
        host: addresses[i].address,
        family: addresses[i].family
      });
      socket.on('error', onError);
      socket.on('connect', onConnect);
    }
    signal.addEventListener('abort', onAbort, {
      once: true
    });
  });
}
async function connectInSequence(options, lookup, signal) {
  if (signal.aborted) {
    throw new _abortError.default();
  }
  const errors = [];
  const addresses = await lookupAllAddresses(options.host, lookup, signal);
  for (const address of addresses) {
    try {
      return await new Promise((resolve, reject) => {
        const socket = _net.default.connect({
          ...options,
          host: address.address,
          family: address.family
        });
        const onAbort = () => {
          socket.removeListener('error', onError);
          socket.removeListener('connect', onConnect);
          socket.destroy();
          reject(new _abortError.default());
        };
        const onError = err => {
          signal.removeEventListener('abort', onAbort);
          socket.removeListener('error', onError);
          socket.removeListener('connect', onConnect);
          socket.destroy();
          reject(err);
        };
        const onConnect = () => {
          signal.removeEventListener('abort', onAbort);
          socket.removeListener('error', onError);
          socket.removeListener('connect', onConnect);
          resolve(socket);
        };
        signal.addEventListener('abort', onAbort, {
          once: true
        });
        socket.on('error', onError);
        socket.on('connect', onConnect);
      });
    } catch (err) {
      if (err instanceof Error && err.name === 'AbortError') {
        throw err;
      }
      errors.push(err);
      continue;
    }
  }
  throw new AggregateError(errors, 'Could not connect (sequence)');
}

/**
 * Look up all addresses for the given hostname.
 */
async function lookupAllAddresses(host, lookup, signal) {
  if (signal.aborted) {
    throw new _abortError.default();
  }
  if (_net.default.isIPv6(host)) {
    return [{
      address: host,
      family: 6
    }];
  } else if (_net.default.isIPv4(host)) {
    return [{
      address: host,
      family: 4
    }];
  } else {
    return await new Promise((resolve, reject) => {
      const onAbort = () => {
        reject(new _abortError.default());
      };
      signal.addEventListener('abort', onAbort);
      const domainInASCII = _nodeUrl.default.domainToASCII(host);
      lookup(domainInASCII === '' ? host : domainInASCII, {
        all: true
      }, (err, addresses) => {
        signal.removeEventListener('abort', onAbort);
        err ? reject(err) : resolve(addresses);
      });
    });
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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