import { PrismaClient } from '@prisma/pg/client';
import fs from 'fs';
const prisma = new PrismaClient();

function getCurrentUtcTime(): string {
    return new Date().toISOString();
}

async function findUserIds(users: string[], company_id: number): Promise<number[]> {
    try {
        const userIds = await prisma.$queryRaw<{ user_id: number }[]>`
            SELECT user_id 
            FROM user_details 
            WHERE company_id = ${company_id} 
            AND CONCAT(first_name, ' ', last_name) IN (${users.join(', ')})
        `;
        return userIds.map(user => user.user_id);
    } catch (error: any) {
        console.error('Error fetching user IDs:', error.message);
        return [];
    }
}

async function findRoleIds(roles: string[], company_id: number): Promise<number[]> {
    try {
        const roleIds = await prisma.role.findMany({
            where: { company_id, name: { in: roles } },
            select: { id: true },
        });
        return roleIds.map(role => role.id);
    } catch (error: any) {
        console.error('Error fetching role IDs:', error.message);
        return [];
    }
}

async function createOrUpdatePolicy(policyData: any, company_id: number, created_by: number, created_at: string): Promise<void> {
    try {

        const policy = await prisma.policies.upsert({
            where: { company_id_name: { company_id, name: policyData.name } },
            update: {
                description: policyData.description,
                slug: policyData.slug,
                query_information: policyData.query_information,
                updated_by: created_by,
                updated_at: created_at,
            },
            create: {
                company_id,
                name: policyData.name,
                description: policyData.description,
                slug: policyData.slug,
                query_information: policyData.query_information,
                created_by,
                created_at,
                updated_by: created_by,
                updated_at: created_at
            },
        });

        if (!policy?.id) {
            console.log('Failed to upsert policy');
            return;
        }

        // Batch upsert roles
        if (policyData.roles.length) {
            const roleIds = await findRoleIds(policyData.roles, company_id);
            console.log(roleIds)
            if (roleIds.length) {
                console.log(company_id)
                console.log(policy.id)
                
                await prisma.rolePolicies.createMany({
                    data: roleIds.map(role_id => ({
                        company_id,
                        policy_id: policy.id,
                        role_id,
                    })),
                    skipDuplicates: true, // Skip duplicates if they exist
                });
            } else {
                console.log(`Failed to find role ids for the Roles: ${policyData.roles.join(', ')}`);
            }
        }

        // Batch upsert users
        if (policyData.users.length) {
            const userIds = await findUserIds(policyData.users, company_id);
            if (userIds.length) {
                await prisma.userPolicies.createMany({
                    data: userIds.map(user_id => ({
                        company_id,
                        policy_id: policy.id,
                        user_id,
                    })),
                    skipDuplicates: true, // Skip duplicates if they exist
                });
            } else {
                console.log(`Failed to find user ids for the Users: ${policyData.users.join(', ')}`);
            }
        }

        console.log('Policy upsertion completed.');
    } catch (error: any) {
        console.error(`Policy seeder failed: ${policyData.name}`, error.message);
        throw error;
    }
}

export async function policiesSeeder(company_id: number): Promise<void> {
    try {
        const policies = JSON.parse(fs.readFileSync(__dirname + '/json/policy-seeds.json', 'utf-8'));
        const user = await prisma.user.findFirst({
            where: { company_id, role: 'super_admin' },
            select: { id: true },
        });

        if (!user) {
            console.log(`Menu seeder failed: admin user not available for company id ${company_id}`);
            return;
        }

        const utcNow = getCurrentUtcTime();
        for (const policyData of policies) {
            await createOrUpdatePolicy(policyData, company_id, user.id, utcNow);
        }

        console.log('Policies upsertion completed.');
    } catch (error: any) {
        console.error('Policies seeder failed:', error.message);
        throw error;
    }
}