import { PrismaClient} from '@prisma/sql/client';
import fs from 'fs';
import path from 'path';

const prisma = new PrismaClient();

export async function gridSeeder(company_id: number) {
  try {
    console.log("🚀 Starting Grid Seeder...");

    const fieldTypes = JSON.parse(
      fs.readFileSync(path.join(__dirname, 'json/field-types.json'), 'utf-8')
    );
    const masterEntitys = JSON.parse(
      fs.readFileSync(path.join(__dirname, 'json/master-grid.json'), 'utf-8')
    );

    for (const masterEntity of masterEntitys) {
      const { export_template_id, lineitems, ...masterEntityData } = masterEntity;

      let export_template: any = null;
      if (export_template_id) {
        const template: any = await prisma.exportTemplate.findFirst({
          where: {
            company_id: company_id,
            slug: export_template_id,
          },
        });
        export_template = template?.id || null;
      }

      // ✅ Upsert MasterEntity
      let mg = await prisma.masterEntity.findFirst({
        where: { company_id, entity_name: masterEntity.entity_name },
      });

      if (mg) {
        mg = await prisma.masterEntity.update({
          where: { id: mg.id },
          data: { 
            ...Object.fromEntries(
              Object.entries(masterEntityData).map(([key, value]) => [
                key,
                typeof value === "object" && value !== null ? JSON.stringify(value) : value,
              ])
            ), 
            export_template_id: export_template 
          },
        });
        console.log(`🔄 Updated MasterEntity: ${masterEntity.entity_name}`);
      } else {
        mg = await prisma.masterEntity.create({
          data: { 
            ...Object.fromEntries(
              Object.entries(masterEntityData).map(([key, value]) => [
                key,
                typeof value === "object" && value !== null ? JSON.stringify(value) : value,
              ])
            ),
            name: masterEntityData.name ?? null, // Ensure `name` is provided
            entity_name: masterEntityData.entity_name ?? null, // Ensure `entity_name` is provided
            entity_type: masterEntityData.entity_type ?? null, // Ensure `entity_type` is provided
            primary_table: masterEntityData.primary_table ?? null, // Ensure `primary_table` is provided
            export_template_id: export_template,
            company_id 
          },
        });
        console.log(`✅ Created MasterEntity: ${masterEntity.entity_name}`);
      }

      if (!mg) continue; // Skip if master entity couldn't be created

      // ✅ Upsert MasterEntityLineItems
      const fieldTypeLength = fieldTypes?.length || 1;
      if (lineitems && lineitems.length) {
        for (const lineitem of lineitems) {
          const field_type_id = lineitem.field_type_id + (company_id - 1) * fieldTypeLength;

          let existingLineItem = await prisma.masterEntityLineItem.findFirst({
            where: {
              company_id,
              master_grid_id: mg.id,
              field_name: lineitem.field_name,
              display_name: lineitem.display_name,
            },
          });

          if (existingLineItem) {
            await prisma.masterEntityLineItem.update({
              where: { id: existingLineItem.id },
              data: { ...lineitem, company_id, master_grid_id: mg.id, field_type_id },
            });
            console.log(`🔄 Updated LineItem: ${lineitem.display_name}`);
          } else {
            await prisma.masterEntityLineItem.create({
              data: { ...lineitem, company_id, master_grid_id: mg.id, field_type_id },
            });
            console.log(`✅ Created LineItem: ${lineitem.display_name}`);
          }
        }
      }
    }

    console.log("✅ Grid Seeder completed successfully.");
  } catch (error) {
    console.error(`❌ Grid seeder failed: ${error}`);
    throw error;
  } finally {
    await prisma.$disconnect();
  }
}
