import { PrismaClient, Prisma } from '@prisma/sql/client';

const prisma = new PrismaClient();

export async function userRolesSeeds(companyId: number) {
  try {
    console.log("🚀 Starting User Roles Seeder...");

    const userRoleMapping: Record<string, string[]> = {
      "mukesh": ['Manager'],
      "raj": ['Supervisor'],
      "mukeshmtechcedence": ['Support Agent', 'Supervisor'],
    };

    const roles = await prisma.role.findMany({
      where: {
        company_id: companyId,
        name: { in: Object.values(userRoleMapping).flat() },
      },
      select: { id: true, name: true },
    });

    const roleNameToIdMap = roles.reduce((map, role) => {
      map[role.name] = role.id;
      return map;
    }, {} as Record<string, number>);

    const superAdminOrAdminUser = await prisma.user.findFirst({
      where: {
        company_id: companyId,
        OR: [{ role: 'super_admin' }, { role: 'admin' }],
      },
    });

    if (!superAdminOrAdminUser) {
      console.log(`❌ User role seeder failed: No admin user found for company_id ${companyId}`);
      return false;
    }

    for (const [username, roleNames] of Object.entries(userRoleMapping)) {
      const user = await prisma.user.findFirst({
        where: {
          company_id: companyId,
          username,
        },
      });

      if (!user) {
        console.error(`❌ Skipping: User ${username} does not exist for company_id ${companyId}`);
        continue;
      }

      for (const roleName of roleNames) {
        const roleId = roleNameToIdMap[roleName];

        if (!roleId) {
          console.error(`❌ Skipping: Invalid role name ${roleName} for user ${username}`);
          continue;
        }

        const existingRole = await prisma.userRoles.findFirst({
          where: {
            company_id: companyId,
            user_id: user.id,
            role_id: roleId,
          },
        });

        if (!existingRole) {
          await prisma.userRoles.create({
            data: {
              company_id: companyId,
              user_id: user.id,
              role_id: roleId,
            },
          });
          console.log(`✅ Created role assignment: Company ${companyId}, Role ${roleName}, User ${username}`);
        } else {
          console.log(`🔄 Role assignment already exists: Company ${companyId}, Role ${roleName}, User ${username}`);
        }
      }
    }

    console.log("✅ User Roles Seeder completed successfully.");
  } catch (error) {
    console.error(`❌ Error in userRolesSeeds: ${error}`);
    if (error instanceof Prisma.PrismaClientKnownRequestError && error.code === 'P2003' && error.meta) {
      console.error(`Foreign key constraint failed: ${error.meta.field_name}`);
    }
    throw error;
  } finally {
    await prisma.$disconnect();
  }
}
