import { Permissions, PrismaClient } from '@prisma/pg/client';
import fs from 'fs';
const prisma = new PrismaClient();
const action_types: any = {
  view: 'view',
  add: 'add',
  edit: 'edit',
  popup_add: 'popup_add',
  popup_edit: 'popup_edit',
  delete: 'delete',
  details: 'details',
  popup_details: 'popup_details',
  export_excel: 'export_excel',
  import: 'import',
  assign: 'assign',
  email_resend: 'email_resend',
  generate_vector: 'generate_vector',
  child_details: 'child_details',
  print: 'print',
};
function getCurrentUtcTime() {
  return new Date().toISOString();
}

async function getMasterEntityIdBySlug(slug: string, company_id: number) {
  const masterEntity = await prisma.masterEntity.findFirst({
    where: {
      company_id,
      entity_name: slug,
    },
  });
  return masterEntity ? masterEntity.id : null;
}

async function findOrCreatePermission(permissionData: any, company_id: number) {
  try {
    const utcNow = getCurrentUtcTime();
    let { rights, ...roleData } = permissionData;

    let order_no = 1;
    for (const right of rights) {
      const slug = action_types[right] + '_' + roleData.slug;

      const masterEntityId = await getMasterEntityIdBySlug(
        roleData.slug,
        company_id,
      );

      await prisma.permissions.upsert({
        where: {
          company_id_slug: {
            // this should match the name of your composite unique constraint
            company_id,
            slug,
          },
        },
        update: { slug, company_id, entity_id: masterEntityId },
        create: {
          slug,
          company_id,
          name: right,
          entity_id: masterEntityId,
          order_no,
        },
      });
      order_no++;
    }

    console.log('Permission creation completed.');
  } catch (error) {
    console.log(`Permission seeder failed: ${permissionData.name}`);
    throw error;
  } finally {
    // await prisma.$disconnect();
  }
}

export async function permissionsSeeder(company_id: number) {
  try {
    const permissions = JSON.parse(
      fs.readFileSync(__dirname + '/json/permission-seeds.json', 'utf-8'),
    );

    for (const permissionData of permissions) {
      await findOrCreatePermission(permissionData, company_id);
    }

    console.log('Permissions creation completed.');
  } catch (error) {
    throw error;
  }
}
