import { PrismaClient } from '@prisma/pg/client';
import fs from 'fs';
import path from 'path';

const prisma = new PrismaClient();

type ThemeLineItemSeed = {
  attribute: string;
  value: string;
  status_id?: number;
  description?: string | null;
};

type ThemeSeed = {
  name: string;
  description?: string | null;
  reference_images?: string | null; // matches String? in your schema
  status_id?: number;
  themeLineItems?: ThemeLineItemSeed[];
  line_items?: ThemeLineItemSeed[];
};

function readThemes(jsonPath: string): ThemeSeed[] {
  const raw = fs.readFileSync(jsonPath, 'utf-8');
  const parsed = JSON.parse(raw);
  if (Array.isArray(parsed)) return parsed as ThemeSeed[];
  if (Array.isArray((parsed as any)?.themes)) return (parsed as any).themes as ThemeSeed[];
  throw new Error(`Invalid theme JSON at ${jsonPath}. Expected an array or { "themes": [...] }`);
}

export async function themeAndLineItemsSeeder(
  company_id: number,
  jsonPath = path.join(__dirname, 'json', 'theme-seeds.json'),
) {
  // Find super_admin to stamp created_by / updated_by
  const adminUser = await prisma.user.findFirst({
    where: { company_id, role: 'super_admin' },
    select: { id: true },
  });
  if (!adminUser) {
    console.log(`Theme seeder failed: super_admin not found for company_id ${company_id}`);
    return false;
  }

  const themes = readThemes(jsonPath);

  for (const raw of themes) {
    const lineItems = (raw.themeLineItems ?? raw.line_items ?? []) as ThemeLineItemSeed[];

    await prisma.$transaction(async (tx) => {
      // ---- THEME: find-or-create, then update ----
      let theme = await tx.themes.findFirst({
        where: { company_id, name: raw.name },
        orderBy: { id: 'asc' }, // pick a canonical row if multiples exist
        select: { id: true, name: true },
      });

      if (!theme) {
        theme = await tx.themes.create({
          data: {
            company_id,
            name: raw.name,
            description: raw.description ?? null,
            reference_images: raw.reference_images ?? null,
            status_id: raw.status_id ?? 1,
            created_by: adminUser.id,
            updated_by: adminUser.id,
            // created_at/updated_at handled by defaults/@updatedAt
          },
          select: { id: true, name: true },
        });
        console.log(`Theme created: ${theme.name}`);
      } else {
        await tx.themes.update({
          where: { id: theme.id },
          data: {
            description: raw.description ?? null,
            reference_images: raw.reference_images ?? null,
            status_id: raw.status_id ?? 1,
            updated_by: adminUser.id,
          },
        });
        console.log(`Theme updated: ${theme.name}`);
      }

      // ---- LINE ITEMS: find-or-create, then update ----
      for (const li of lineItems) {
        const existingLI = await tx.themeLineItems.findFirst({
          where: {
            company_id,
            theme_id: theme.id,
            attribute: li.attribute,
          },
          orderBy: { id: 'asc' }, // canonical row if duplicates exist
          select: { id: true },
        });

        if (!existingLI) {
          await tx.themeLineItems.create({
            data: {
              company_id,
              theme_id: theme.id,
              attribute: li.attribute,
              value: li.value,
              description: li.description ?? null,
              status_id: li.status_id ?? 1,
              created_by: adminUser.id,
              updated_by: adminUser.id,
              // timestamps auto
            },
          });
          console.log(`  • LineItem created: ${li.attribute}`);
        } else {
          await tx.themeLineItems.update({
            where: { id: existingLI.id },
            data: {
              value: li.value,
              description: li.description ?? null,
              status_id: li.status_id ?? 1,
              updated_by: adminUser.id,
            },
          });
          console.log(`  • LineItem updated: ${li.attribute}`);
        }
      }
    });
  }

  console.log('Theme + Line Items seeding completed.');
  return true;
}
