import { PrismaClient } from '@prisma/sql/client';
import fs from 'fs';
import path from 'path';

const prisma = new PrismaClient();

export async function importTemplatesSeeder(company_id: number) {
  try {
    console.log("🚀 Starting Import Template Seeder...");

    const importTemplates = JSON.parse(
      fs.readFileSync(path.join(__dirname, 'json/import-templates.json'), 'utf-8')
    );

    for (const importTemplate of importTemplates) {
      const { lineitems, queries, ...importTemplateData } = importTemplate;

      let user = await prisma.user.findFirst({
        where: { company_id, role: 'super_admin' },
      });

      if (!user) {
        console.log(`❌ Import template seeder failed: No super_admin for company_id ${company_id}`);
        return false;
      }

      // ✅ Upsert Import Template
      let mg = await prisma.importTemplate.findFirst({
        where: { company_id, slug: importTemplateData.slug },
      });

      if (mg) {
        mg = await prisma.importTemplate.update({
          where: { id: mg.id },
          data: {
            ...importTemplateData,
            updated_by: user.id,
            updated_at: new Date(),
          },
        });
        console.log(`🔄 Updated Import Template: ${importTemplateData.slug}`);
      } else {
        mg = await prisma.importTemplate.create({
          data: {
            ...importTemplateData,
            company_id,
            created_by: user.id,
          },
        });
        console.log(`✅ Created Import Template: ${importTemplateData.slug}`);
      }

      if (!mg) continue; // Skip if import template couldn't be created

      // ✅ Upsert LineItems
      for (const lineitem of lineitems) {
        let field_type = await prisma.fieldType.findFirst({
          where: {
            company_id,
            field_type: lineitem.field_type_id,
          },
        });

        if (!field_type) {
          console.log(`❌ Missing fieldType: ${lineitem.field_type_id} for company_id ${company_id}`);
          return false;
        }

        let existingLineItem = await prisma.importTemplateLineItem.findFirst({
          where: { company_id, import_template_id: mg.id, display_name: lineitem.display_name },
        });

        if (existingLineItem) {
          await prisma.importTemplateLineItem.update({
            where: { id: existingLineItem.id },
            data: {
              ...lineitem,
              import_template_id: mg.id,
              company_id,
              field_type_id: field_type.id,
            },
          });
          console.log(`🔄 Updated LineItem: ${lineitem.display_name}`);
        } else {
          await prisma.importTemplateLineItem.create({
            data: {
              ...lineitem,
              import_template_id: mg.id,
              company_id,
              field_type_id: field_type.id,
            },
          });
          console.log(`✅ Created LineItem: ${lineitem.display_name}`);
        }
      }

      // ✅ Upsert Queries
      for (const query of queries) {
        let existingQuery = await prisma.importTemplateQueries.findFirst({
          where: { company_id, import_template_id: mg.id, query_name: query.query_name },
        });

        if (existingQuery) {
          await prisma.importTemplateQueries.update({
            where: { id: existingQuery.id },
            data: {
              ...query,
              import_template_id: mg.id,
              company_id,
            },
          });
          console.log(`🔄 Updated Query: ${query.query_name}`);
        } else {
          await prisma.importTemplateQueries.create({
            data: {
              ...query,
              import_template_id: mg.id,
              company_id,
            },
          });
          console.log(`✅ Created Query: ${query.query_name}`);
        }
      }
    }

    console.log("✅ Import Template Seeder completed successfully.");
  } catch (error) {
    console.error(`❌ Import template seeder failed: ${error}`);
    throw error;
  } finally {
    await prisma.$disconnect();
  }
}
