

import { PrismaClient } from '@prisma/pg/client'; // your import
import * as fs from 'fs/promises';
import { lstat } from 'fs/promises';



import * as path from 'path';
import * as fsSync from 'fs';
import * as dotenv from 'dotenv';

// Allow explicit --env override
const args = process.argv.slice(2);
const get = (flag: string) => {
  const i = args.findIndex(a => a === `--${flag}` || a.startsWith(`--${flag}=`));
  if (i === -1) return null;
  const a = args[i]; const eq = a.indexOf('=');
  return eq > -1 ? a.slice(eq + 1) : null;
};
const explicitEnv = get('env') || process.env.DOTENV_CONFIG_PATH || null;

// Guess env name (development / production / etc.)
const guessedEnv = process.env.ENVIRONMENT || process.env.NODE_ENV || 'development';

const CWD = process.cwd();
const candidates = explicitEnv
  ? [path.resolve(explicitEnv)]
  : [
      // root
      path.join(CWD, `.env.${guessedEnv}`),
      path.join(CWD, `.env.local`),
      path.join(CWD, `.env`),
      // prisma/
      path.join(CWD, 'prisma', `.env.${guessedEnv}`),
      path.join(CWD, 'prisma', `.env`),
      // prisma/pg/my_scripts/ (just in case)
      path.join(CWD, 'prisma', 'pg', 'my_scripts', `.env.${guessedEnv}`),
      path.join(CWD, 'prisma', 'pg', 'my_scripts', `.env`),
    ];

const envPath = candidates.find(p => fsSync.existsSync(p));
if (envPath) {
  dotenv.config({ path: envPath });
  console.log(`[env] Loaded: ${envPath}`);
} else {
  console.warn(`[env] No .env found. Checked: ${candidates.join(' , ')}`);
}

if (!process.env.DATABASE_URL) {
  console.error(
    "[env] DATABASE_URL is not set. Add it to your env file or pass --env=PATH.\n" +
    "Example:\nDATABASE_URL=postgresql://postgres:postgres@localhost:5432/lcp_framework"
  );
  process.exit(1);
}

type Opts = {
  dir: string;
  only: string[];
  except: string[];
  singleTx: boolean;
  dryRun: boolean;
};


function splitPostgresStatements(sql: string): string[] {
    const out: string[] = [];
    let buf = '';
    let i = 0;
  
    let inSingle = false;     // '...'
    let inDouble = false;     // "..."
    let inLineComment = false; // -- ...
    let inBlockComment = false; // /* ... */
    let dollarTag: string | null = null; // $tag$...$tag$ or $$...$$
  
    const isIdent = (c: string) => /[A-Za-z0-9_]/.test(c);
  
    while (i < sql.length) {
      const ch = sql[i];
      const nxt = i + 1 < sql.length ? sql[i + 1] : '';
  
      // handle line comment
      if (inLineComment) {
        buf += ch;
        if (ch === '\n') inLineComment = false;
        i++;
        continue;
      }
      // handle block comment
      if (inBlockComment) {
        buf += ch;
        if (ch === '*' && nxt === '/') {
          buf += nxt; i += 2; inBlockComment = false; continue;
        }
        i++; continue;
      }
      // handle dollar-quoted string
      if (dollarTag) {
        buf += ch;
        // check if we are at the end tag
        if (ch === '$') {
          // try to match the full tag ending at current index
          const tagLen = dollarTag.length;
          const start = i - tagLen + 1;
          if (start >= 0 && sql.slice(start, i + 1) === dollarTag) {
            dollarTag = null; // closed
          }
        }
        i++; continue;
      }
      // handle regular quotes
      if (inSingle) {
        buf += ch;
        if (ch === '\'' && nxt === '\'') { // escaped '' inside string
          buf += nxt; i += 2; continue;
        }
        if (ch === '\'') inSingle = false;
        i++; continue;
      }
      if (inDouble) {
        buf += ch;
        if (ch === '"' && nxt === '"') { // escaped "" inside identifier
          buf += nxt; i += 2; continue;
        }
        if (ch === '"') inDouble = false;
        i++; continue;
      }
  
      // we are at top-level code (not inside quotes/comments/dollar)
  
      // start of line comment?
      if (ch === '-' && nxt === '-') {
        buf += ch + nxt; i += 2; inLineComment = true; continue;
      }
      // start of block comment?
      if (ch === '/' && nxt === '*') {
        buf += ch + nxt; i += 2; inBlockComment = true; continue;
      }
      // start of single quote?
      if (ch === '\'') { buf += ch; inSingle = true; i++; continue; }
      // start of double quote?
      if (ch === '"') { buf += ch; inDouble = true; i++; continue; }
      // start of dollar-quote? $ or $tag$
      if (ch === '$') {
        // find a tag like $tag$ or just $$
        let j = i + 1;
        while (j < sql.length && isIdent(sql[j])) j++;
        if (j < sql.length && sql[j] === '$') {
          const tag = sql.slice(i, j + 1); // includes both $
          dollarTag = tag;
          buf += tag;
          i = j + 1;
          continue;
        }
        // lone $ that isn't a tag
        buf += ch; i++; continue;
      }
  
      // statement terminator at top-level
      if (ch === ';') {
        const stmt = buf.trim();
        if (stmt.length) out.push(stmt + ';'); // keep semicolon per stmt
        buf = '';
        i++;
        continue;
      }
  
      // default
      buf += ch;
      i++;
    }
  
    const tail = buf.trim();
    if (tail.length) out.push(tail); // last statement may be missing semicolon
    // strip empty/comment-only statements
    return out.filter(s => /\S/.test(s));
  }

  
function parseArgs(): Opts {
  const args = process.argv.slice(2);
  const get = (flag: string) => {
    const i = args.findIndex(a => a === `--${flag}` || a.startsWith(`--${flag}=`));
    if (i === -1) return null;
    const a = args[i];
    const eq = a.indexOf('=');
    return eq > -1 ? a.slice(eq + 1) : null;
  };

  //const dir = get('dir') || path.join(process.cwd(), 'db', 'procedures');
  console.log(get('dir'))
  const dir = get('dir') || path.join(process.cwd(), 'prisma', 'pg', 'my_scripts', 'db-run', 'common');

  const parsePatterns = (s: string | null) =>
    (s || '').split(',').map(x => x.trim()).filter(Boolean);

  const only = parsePatterns(get('only'));
  const except = parsePatterns(get('except'));
  const singleTx = args.includes('--single-tx');
  const dryRun = args.includes('--dry-run');

  return { dir, only, except, singleTx, dryRun };
}

function wildcardToRegex(pat: string): RegExp {
  const esc = pat
    .replace(/[-/\\^$+.,()|[\]{}]/g, '\\$&')
    .replace(/\*/g, '.*')
    .replace(/\?/g, '.');
  return new RegExp(`^${esc}$`, 'i');
}
function matchAny(name: string, patterns: string[]): boolean {
  if (!patterns.length) return false;
  return patterns.some(p => wildcardToRegex(p).test(name));
}

// Block psql meta-commands like \i, \set, etc. (Prisma can't execute them)
function containsPsqlMeta(sql: string): boolean {
  return sql.split(/\r?\n/).some(line => /^\s*\\[a-z]/i.test(line));
}

async function listSqlFiles(baseDir: string): Promise<string[]> {
  const entries = await fs.readdir(baseDir);
  const sqls: string[] = [];
  for (const name of entries) {
    if (!name.toLowerCase().endsWith('.sql')) continue;

    const full = path.join(baseDir, name);
    const st = await lstat(full);

    if (!st.isFile() || st.isSymbolicLink()) continue; // no dirs/symlinks

    const rp = path.resolve(full);
    if (!rp.startsWith(baseDir + path.sep)) continue;   // path breakout guard

    sqls.push(name);
  }
  // natural sort (001_, 010_, …)
  sqls.sort((a, b) => a.localeCompare(b, undefined, { numeric: true, sensitivity: 'base' }));
  return sqls;
}

type PrismaExec = { $executeRawUnsafe: (query: string) => Promise<number> };

async function main() {
  const { dir, only, except, singleTx, dryRun } = parseArgs();
  const baseDir = path.resolve(dir);

  // dir must exist
  try {
    const st = await lstat(baseDir);
    if (!st.isDirectory()) throw new Error(`${baseDir} is not a directory`);
  } catch (e: any) {
    console.error(`Directory not accessible: ${baseDir}\n${e?.message || e}`);
    process.exit(1);
  }

  let files = await listSqlFiles(baseDir);
  if (only.length)   files = files.filter(f => matchAny(f, only));
  if (except.length) files = files.filter(f => !matchAny(f, except));

  console.log(`Directory: ${baseDir}`);
  console.log(`Selected ${files.length} file(s):`);
  files.forEach(f => console.log(`  - ${f}`));

  if (!files.length) {
    console.log('Nothing to do.');
    return;
  }

  if (dryRun) {
    console.log('Dry run: not executing SQL.');
    return;
  }

  const prisma = new PrismaClient();

  const runFile = async (exec: PrismaExec, file: string) => {
    const full = path.join(baseDir, file);
    const sql = await fs.readFile(full, 'utf8');
  
    if (!sql.trim()) {
      console.log(`(skip empty) ${file}`);
      return;
    }
    if (containsPsqlMeta(sql)) {
      throw new Error(
        `psql meta-commands (\\*) detected in ${file}. ` +
        `Prisma cannot execute them. Use a psql-based runner or remove meta-commands.`
      );
    }
  
    const statements = splitPostgresStatements(sql);
    console.log(`Executing: ${file} (${statements.length} statement${statements.length === 1 ? '' : 's'})`);
  
    for (let idx = 0; idx < statements.length; idx++) {
      const stmt = statements[idx].trim();
      if (!stmt) continue;
      try {
        await exec.$executeRawUnsafe(stmt);
      } catch (e: any) {
        console.error(`  ✖ Failed at statement #${idx + 1}:\n${stmt}\n  → ${e.message}`);
        throw e;
      }
    }
    console.log(`✔ ${file}`);
  };
  

  try {
    if (singleTx) {
      console.log('Running all files in a single transaction...');
      await prisma.$transaction(async (tx) => {
        for (const f of files) {
          await runFile(tx as unknown as PrismaExec, f);
        }
      });
      console.log('All done.');
    } else {
      // per-file transactions
      for (const f of files) {
        try {
          await prisma.$transaction(async (tx) => {
            await runFile(tx as unknown as PrismaExec, f);
          });
        } catch (e: any) {
          console.error(`✖ ${f} failed: ${e.message}`);
          process.exit(1);
        }
      }
      console.log('All done.');
    }
  } finally {
    await prisma.$disconnect();
  }
}

main().catch(err => {
  console.error('Failed:', err);
  process.exit(1);
});
