import { PrismaClient } from '@prisma/pg/client';
import fs from 'fs';
const prisma = new PrismaClient();

export async function TeamSeeder(company_id: number) {
  try {
    const rows = JSON.parse(
      fs.readFileSync(__dirname + '/json/team-seeds.json', 'utf-8'),
    );

    const user = await prisma.user.findFirst({
      where: { company_id: company_id, role: 'super_admin' },
    });

    if (!user?.id) {
      throw new Error('Super admin user not found!');
    }

    for (const row of rows) {
      // Check if the Team already exists by its name and company_id
      const existingTeam = await prisma.teams.findUnique({
        where: { company_id_name: { company_id, name: row.name } },
      });

      if (existingTeam) {
        // Update the Team if it already exists
        await prisma.teams.update({
          where: { id: existingTeam.id },
          data: {
            name: row.name,
            status_id: row.status_id || 1, // Default to active status if not provided
            updated_by: user.id, // Ensure updated_by is assigned
          },
        });
        console.log(`Updated Team with name ${row.name}`);
      } else {
        // Create a new Team if it doesn't exist
        await prisma.teams.create({
          data: {
            name: row.name,
            company_id,
            status_id: row.status_id || 1, // Default to active status if not provided
            created_by: user.id, // Ensure created_by is assigned
            updated_by: user.id, // Ensure updated_by is assigned
          },
        });
        console.log(`Created Team with name ${row.name}`);
      }
    }
  } catch (error) {
    console.error(`Error in TeamSeeder: ${error}`);
    throw error;
  } finally {
    // await prisma.$disconnect();
  }
}
