import { PrismaClient, ApproverType, Prisma } from '@prisma/pg/client';

const prisma = new PrismaClient();

function getCurrentUtcTime() {
  return new Date().toISOString();
}

export async function approvalWorkflow(company_id: number) {
  console.log("Seeding ApprovalWorkflow...");
  
  let user = await prisma.user.findFirst({
    where: { company_id: company_id, role: 'super_admin' }
  });
  if (!user) {
      console.log(`Department seeder failed: admin user not available for company id ${company_id}`);
      return false;
  }

  const createdBy = user.id;
  const utcNow = getCurrentUtcTime();

  const approvalWorkflows = [
    {
      slug: "name_starts_with_mukesh",
      name: "name starts with 'Mukesh'",
      description: "name starts with 'Mukesh'",
      query_information: `SELECT count(1) as exist FROM users u LEFT JOIN user_details ud ON ud.user_id = u.id WHERE u.email = $prefill_data.users.email AND u.status_id = 1 AND CONCAT($prefill_data.user_details.first_name, ' ', $prefill_data.user_details.last_name) ILIKE '%Mukesh%'`,
      company_id,
      created_by: createdBy,
      updated_by: createdBy,
      created_at: utcNow,
      updated_at: utcNow,
    },
    {
      slug: "cosmetic_order_item_cat",
      name: "Cosmetic Order Item Category",
      description: "Cosmetic Order Item Category",
      query_information: `SELECT COUNT(1) as exist FROM orders as o 
                          INNER JOIN order_details as od ON od.order_id = o.id 
                          INNER JOIN product as p on p.id = od.item_id
                          WHERE o.id = $unique_id 
                          AND p.category_id = 5`,
      company_id,
      created_by: createdBy,
      updated_by: createdBy,
      created_at: utcNow,
      updated_at: utcNow,
    },
    {
      slug: "high_value_transaction",
      name: "High Value Transaction Approval",
      description: "High Value Transaction Approval",
      query_information: `SELECT COUNT(1) as exist FROM transactions 
                          WHERE amount > 50000 
                          AND user_id = $session_user_id`,
      company_id,
      created_by: createdBy,
      updated_by: createdBy,
      created_at: utcNow,
      updated_at: utcNow,
    },
    {
      slug: "finance_approval_required",
      name: "Finance Approval Required",
      description: "Finance Approval Required",
      query_information: `SELECT COUNT(1) as exist FROM orders 
                          WHERE id = $unique_id 
                          AND payment_status = 'pending' 
                          AND total_amount > 20000`,
      company_id,
      created_by: createdBy,
      updated_by: createdBy,
      created_at: utcNow,
      updated_at: utcNow,
    }
  ];

  // Insert data using upsert (to prevent duplicate entries)
  for (const workflow of approvalWorkflows) {
    await prisma.approvalWorkflow.upsert({
      where: { company_id_slug : {company_id, slug: workflow.slug} },
      update: workflow,
      create: workflow,
    });
    await approvalWorkflowAssignment(company_id, workflow.slug, createdBy, utcNow);
  }

  console.log("ApprovalWorkflow seeding completed!");
}

async function approvalWorkflowAssignment(company_id: number, slug:string, created_by: number, utcNow: string) {
    console.log("Seeding ApprovalWorkflowAssignment...");
  
    // Fetch existing approval workflows by slug
    const workFlow = await prisma.approvalWorkflow.findUnique({
      where: {company_id_slug : { company_id, slug }},
    });
  
    if (workFlow) {
        const approvalAssignments = [
          {
              approval_workflow_id: workFlow.id,
              approver_order_no: 1,
              approver_type: ApproverType.user_id,
              approver: "1",
              approve_mail_id: 9,
              reject_mail_id: 8,
              // approve_query_information: Prisma.JsonNull,
              // reject_query_information: Prisma.JsonNull,
              company_id,
              created_by: created_by,
              updated_by: created_by,
              created_at: utcNow,
              updated_at: utcNow,
          },
          {
              approval_workflow_id: workFlow.id,
              approver_order_no: 2,
              approver_type: ApproverType.user_id,
              approver: "1,2",
              // approve_query_information: Prisma.JsonNull,
              // reject_query_information: Prisma.JsonNull,
              company_id,
              created_by: created_by,
              updated_by: created_by,
              created_at: utcNow,
              updated_at: utcNow,
          },
          {
              approval_workflow_id: workFlow.id,
              approver_order_no: 3,
              approver_type: ApproverType.role_id,
              approver: "4",
              // approve_query_information: Prisma.JsonNull,
              // reject_query_information: Prisma.JsonNull,
              company_id,
              created_by: created_by,
              updated_by: created_by,
              created_at: utcNow,
              updated_at: utcNow,
          }
        ];
    
        // Insert data using upsert (to prevent duplicate entries)
        for (const assignment of approvalAssignments) {
        await prisma.approvalWorkflowAssignment.upsert({
              where: {
                company_id_approval_workflow_id_approver_order_no: {
                    company_id: assignment.company_id,
                    approval_workflow_id: assignment.approval_workflow_id,
                    approver_order_no: assignment.approver_order_no,
                }
              },
              update: assignment,
              create: assignment,
          });
        }
    }
    console.log("ApprovalWorkflowAssignment seeding completed!");
  }

/*
#INSERT
INSERT INTO approval_process_jobs 
(approval_workflow_slug, name, primary_table, unique_id, user_id, prefill_data, company_id, status_id) 
VALUES 
(
  'order_greaterthan_10000', 
  'Order > 10000 Approval', 
  'orders', 
  1, 
  1, 
  '{}'::jsonb, -- Empty JSON object instead of NULL
  1,
  1 -- Default status
)
ON CONFLICT (company_id, id) DO NOTHING;

#UPDATE
UPDATE approval_process_jobs 
SET 
  name = 'Order > 10000 Approval',
  primary_table = 'orders',
  unique_id = 1,
  user_id = 1,
  prefill_data = '{}'::jsonb, -- Empty JSON object instead of NULL
  status_id = 1
WHERE 
  approval_workflow_slug = 'order_greaterthan_10000'
  AND company_id = 1;
*/