import { PrismaClient } from '@prisma/pg/client';
import fs from 'fs';
const prisma = new PrismaClient();

export async function carouselTemplatesSeeder(company_id: number) {
  try {
    const carouselTemplates = JSON.parse(
      fs.readFileSync(__dirname + '/json/carousel-templates.json', 'utf-8'),
    );

    for (const carouselTemplate of carouselTemplates) {
      const { lineitems, queries, ...carouselTemplateData } = carouselTemplate;

      let user = await prisma.user.findFirst({
        where: { company_id, role: 'super_admin' },
      });
      if (!user) {
        console.log(
          `Carousel template seeder failed: admin user not available for company id ${company_id}`,
        );
        return false;
      }

      // Transaction: Delete existing template (and its line items) if exists, then create new
      await prisma.$transaction(async (tx) => {
        // Find existing template by slug
        const existing = await tx.carouselTemplate.findUnique({
          where: {
            company_id_slug: {
              company_id,
              slug: carouselTemplateData.slug,
            },
          },
        });
        if (existing) {
          // Delete line items first (if not ON DELETE CASCADE)
          await tx.carouselTemplateLineItem.deleteMany({
            where: {
              company_id,
              carousel_template_id: existing.id,
            },
          });
          // Delete the template
          await tx.carouselTemplate.delete({
            where: {
              company_id_slug: {
                company_id,
                slug: carouselTemplateData.slug,
              },
            },
          });
        }

        // Create new Carousel Template
        const mg = await tx.carouselTemplate.create({
          data: {
            ...carouselTemplateData,
            company_id,
            created_by: user.id,
          },
        });

        // Create LineItems
        for (const lineitem of lineitems) {
          let field_type = await tx.fieldType.findFirst({
            where: {
              company_id,
              field_type: {
                equals: lineitem.field_type_id,
                mode: 'insensitive',
              },
            },
          });
          if (!field_type) {
            console.log(
              `Carousel template seeder failed: ${lineitem.field_type_id} field_type not available for company id ${company_id}`,
            );
            throw new Error('Field type not found');
          }

          await tx.carouselTemplateLineItem.create({
            data: {
              ...lineitem,
              carousel_template_id: mg.id,
              company_id,
            },
          });
        }
      });
    }

    console.log('Carousel template creation completed.');
  } catch (error) {
    console.log(`Carousel template seeder failed.`);
    throw error;
  }
}
