import { PrismaClient } from '@prisma/pg/client';

const prisma = new PrismaClient();

function getCurrentUtcTime() {
  return new Date().toISOString();
}

export async function departmentSeeds(company_id: number) {
  try {
    let user = await prisma.user.findFirst({
      where: { company_id: company_id, role: 'super_admin' },
    });
    if (!user) {
      console.log(
        `Department seeder failed: admin user not available for company id ${company_id}`,
      );
      return false;
    }

    const createdBy = user.id;
    const utcNow = getCurrentUtcTime();
    const departments: any[] = [];
    const codes = departments.map((dept: any) => dept.code);
    // Fetch existing departments for the company
    const existingDepartments = await prisma.department.findMany({
      where: { company_id, code: { in: codes } },
    });

    const existingDepartmentNames = new Set(
      existingDepartments.map((dept) => dept),
    );

    const deptPromises: Promise<void>[] = [];

    for (let dept of departments) {
      if (!existingDepartmentNames.has(dept.code)) {
        deptPromises.push(
          prisma.department
            .create({
              data: dept,
            })
            .then((dept) => {
              console.log(
                `Department ${dept.name} created successfully. id is ${dept.id}`,
              );
            })
            .catch((error) => {
              console.error(
                `Failed to create department ${dept.name} for user_id ${createdBy}:`,
                error,
              );
            }),
        );
      } else {
        console.log(
          `Department ${dept.name} already exists for company_id ${company_id}, skipping creation.`,
        );
      }
    }

    await Promise.all(deptPromises);

    console.log('Department seeding completed.');
  } catch (error) {
    console.error('Department seeding failed:', error);
  } finally {
    // await prisma.$disconnect();
  }
}
