#!/bin/bash

echo "🚀 Starting CSV export process..."

# Input Params
ENVIRONMENT=${1:-development}
EXPORT_PATH=$2
MENU_ITEM_ID=$3
CHILD_PROCESS_SLUG=$4

# Input Validation
if [ -z "$ENVIRONMENT" ] || [ -z "$EXPORT_PATH" ] || [ -z "$MENU_ITEM_ID" ] || [ -z "$CHILD_PROCESS_SLUG" ]; then
  echo "❌ Missing required arguments. Usage:"
  echo "   bash export_excel.sh <env> <file_path> <menu_item_id> <child_process_slug>"
  exit 1
fi

# Environment File
ENV_FILE=".env.${ENVIRONMENT}"
if [ ! -f "$ENV_FILE" ]; then
  echo "❌ Environment file '$ENV_FILE' not found!"
  exit 1
fi

echo "✅ Using environment file: $ENV_FILE"

# Normalize line endings
sed -i 's/\r$//' "$ENV_FILE"

# Load environment vars safely (handle 'key = value' or 'key= 'value' ')
TEMP_ENV_FILE="/tmp/tmp_env_$RANDOM.env"
grep -v '^#' "$ENV_FILE" | sed -E "s/^[[:space:]]*([A-Za-z0-9_]+)[[:space:]]*=[[:space:]]*'?([^']*)'?[[:space:]]*$/\1=\"\2\"/" > "$TEMP_ENV_FILE"

set -a
source "$TEMP_ENV_FILE"
set +a
rm -f "$TEMP_ENV_FILE"

# Validate DB environment
if [ -z "$DATABASE_HOST" ] || [ -z "$DATABASE_PORT" ] || [ -z "$DATABASE_USER" ] || [ -z "$DATABASE_PASSWORD" ] || [ -z "$DATABASE_NAME" ]; then
  echo "❌ One or more database environment variables are missing."
  exit 1
fi

# Compose full DATABASE_URL
DATABASE_URL="postgresql://$DATABASE_USER:$DATABASE_PASSWORD@$DATABASE_HOST:$DATABASE_PORT/$DATABASE_NAME"
echo "📌 DATABASE_URL: $DATABASE_URL"

# Create export folder
mkdir -p "$EXPORT_PATH"

# Unique file name
TIMESTAMP=$(date +"%Y%m%d_%H%M%S")
UUID=$(date +%s%N | md5sum | head -c 12)
FILENAME="${UUID}_${TIMESTAMP}.csv"
FULL_PATH="$EXPORT_PATH/$FILENAME"

# Export data using COPY
echo "📡 Exporting data using \COPY..."

PGPASSWORD=$DATABASE_PASSWORD psql -U "$DATABASE_USER" -d "$DATABASE_NAME" -h "$DATABASE_HOST" -p "$DATABASE_PORT" -c "\COPY (SELECT  \"SYS-LOT_NO\",
    \"MTR-SKU\",
    \"MRN-PARENT_SKU\",
    \"MI-FINAL SKU\",
    \"MI-FINAL_SKU_TYPE\",
    \"MI-DESCRIPTION\",
    \"MTR-UOM\",
    \"MTR-QTY\",
    \"MI-UOM\",
    \"SYS-QTY_ALLOC\",
    \"SYS-BAL_QTY\",
    \"MI-UOM_1\",
    \"SYS-QTY_1\",
    \"MI-UOM_2\",
    \"SYS-QTY_2\",
    \"MTR-SERIAL_NO\",
    \"MTR-ZAIN_TAG\",
    \"MTR-ITEM_CONDITION\",
    \"MI-MANUFACTURER\",
    \"MTR-SITE_ID_ORIGIN\",
    \"MTR-SITE_ID_DESTN\",
    \"SYS-AGING\",
    \"MTR-STORAGE_AGE\",
    \"SYS-AGE_YEAR\",
    \"MRN-WH_REF_NO\",
    \"MTR-VENDOR_NAME\",
    \"MI-DEPARTMENT\",
    \"MI-TEAM\",
    \"MI-OLD WH_ID\",
    \"MRN-NEW_WH_ID\",
    \"MI-FINAL_SKU_DESCRIPTION\",
    \"MRN-NEW_WH_LOCATION\",
    \"MI-FINAL_SKU_UOM\",
    \"MTR-REGION\",
    \"MTR-ACTION\",
    \"MTR-ITEM_TO_BE_REUSED\",
    \"MTR-REQUESTOR_NAME\",
    \"SYS-ADD_DATE\",
    \"SYS-EDIT_DATE\",
    \"MTR-REMARKS\",
    \"MI-ITEM_TYPE\",
    \"MRN-PALLETID\",
    \"MRN-PARENT_TAG\",
    \"MTR-MTR_SEQ\",
    \"MTR-AMU_RESPONSE\",
    \"MTR-AMU_REF\",
    \"MTR-AMU_NBV\",
    \"MRN-WH_REMARKS\",
    \"SYS-TAG-STATUS\",
    \"MTR-PO_NUMBER\",
    \"MTR-PROJECT_NAME\",
    \"MTR-ZAIN_PROJECT_OWNER\",
    \"MTR-ITEM_RELATED_TO_TEAM\",
    \"MTR-MAIN_SUB\",
    \"MTR-MAIN_SEQ\",
    \"MI-ITEM_DIMENSION\",
    \"MTR-NET_REFERRENCE\",
    \"STATUS\" FROM stocks_export_view) TO '$FULL_PATH' CSV HEADER"

if [ $? -ne 0 ] || [ ! -f "$FULL_PATH" ]; then
  echo "❌ Failed to export data using \COPY."
  exit 1
fi

echo "✅ CSV file saved to: $FULL_PATH"

# Update in database
echo "📝 Updating child_processes.downloadables in DB..."

PGPASSWORD=$DATABASE_PASSWORD psql -h "$DATABASE_HOST" -U "$DATABASE_USER" -d "$DATABASE_NAME" -p "$DATABASE_PORT" <<EOF

UPDATE child_processes
SET downloadables = '$FULL_PATH'
WHERE slug = '$CHILD_PROCESS_SLUG';


INSERT INTO child_process_logs (
    child_process_id,
    downloadables,
    process_status,
    output,
    initiated_by,
    created_at
)
SELECT
    id,
    '$FULL_PATH',
    'success',  
    'success',  
    1,          
    NOW()       
FROM
    child_processes
WHERE
    slug = '$CHILD_PROCESS_SLUG';
EOF

echo "✅ CSV file saved to: $FULL_PATH"
echo "<link>$FULL_PATH<link>"


if [ $? -eq 0 ]; then
  echo "✅ child_processes.downloadables updated for slug: $CHILD_PROCESS_SLUG"
else
  echo "❌ Failed to update child_processes table."
fi
